
const STORE = {
  BASE: "base_origin",
  USER: "auth_user",
  XTOKEN: "x_token",
  DEVICES: "devices",
  AUTOCALL: "auto_call"
};
const DEFAULT_BASE = "http://45.252.248.148:8010";

const $ = id => document.getElementById(id);

async function load(){
  const cfg = await chrome.storage.sync.get({
    [STORE.BASE]: DEFAULT_BASE,
    [STORE.USER]: "",
    [STORE.XTOKEN]: "",
    [STORE.DEVICES]: [],
    [STORE.AUTOCALL]: false
  });
  $("auth_user").value = cfg[STORE.USER] || "";
  $("auto_call").checked = !!cfg[STORE.AUTOCALL];
  $("base_origin").value = cfg[STORE.BASE] || DEFAULT_BASE;
  renderDevices(cfg[STORE.DEVICES] || []);
}

function renderDevices(list){
  const tbody = $("devRows");
  tbody.innerHTML = "";
  (list||[]).forEach((d, i) => tbody.appendChild(buildRow(i, d.name||"", d.device_token||"")));
}

function buildRow(idx, name, token){
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input type="text" data-idx="${idx}" data-k="name" value="${name||""}" placeholder="VD: CSKH-Ngân"></td>
    <td><input type="text" data-idx="${idx}" data-k="device_token" value="${token||""}" placeholder="token thiết bị"></td>
    <td style="white-space:nowrap">
      <button data-idx="${idx}" data-act="test">Kiểm tra</button>
      <button data-idx="${idx}" data-act="del" class="ghost">Xoá</button>
    </td>
    <td><span id="st_${idx}" class="status">—</span></td>
  `;
  return tr;
}

function collectDevices(){
  const rows = Array.from(document.querySelectorAll("#devRows tr"));
  const out = rows.map((tr) => {
    const name = tr.querySelector('input[data-k="name"]').value.trim();
    const tok  = tr.querySelector('input[data-k="device_token"]').value.trim();
    return {name, device_token: tok};
  }).filter(d => d.device_token);
  return out;
}

$("btnAdd").addEventListener("click", () => {
  const tbody = $("devRows");
  const idx = tbody.querySelectorAll("tr").length;
  tbody.appendChild(buildRow(idx, "", ""));
});

$("devRows").addEventListener("click", async (e) => {
  const btn = e.target.closest("button");
  if (!btn) return;
  const act = btn.dataset.act;
  const idx = Number(btn.dataset.idx);
  const tbody = $("devRows");

  if (act === "del"){
    btn.closest("tr").remove();
    // Re-index ids to keep st_{idx} consistent
    Array.from(tbody.querySelectorAll("tr")).forEach((tr, i) => {
      tr.querySelectorAll("input").forEach(inp => inp.dataset.idx = String(i));
      tr.querySelector('button[data-act="test"]').dataset.idx = String(i);
      tr.querySelector('button[data-act="del"]').dataset.idx  = String(i);
      const st = tr.querySelector(".status");
      st.id = "st_" + i;
    });
    return;
  }

  if (act === "test"){
    await testOneDevice(idx);
    return;
  }
});

async function testOneDevice(idx){
  const row = document.querySelector(`#devRows tr:nth-child(${idx+1})`);
  if (!row) return {ok:false, error:"Row not found"};
  const tok = row.querySelector('input[data-k="device_token"]').value.trim();
  const name = row.querySelector('input[data-k="name"]').value.trim() || `Thiết bị ${idx+1}`;

  // ensure status cell exists
  let st = document.getElementById(`st_${idx}`);
  if (!st){
    const td = row.querySelector("td:last-child");
    if (td){
      st = document.createElement("span");
      st.id = `st_${idx}`; st.className = "status";
      td.innerHTML = ""; td.appendChild(st);
    }
  }
  if (!st) return {ok:false, error:"No status cell"};

  st.textContent = "Đang kiểm tra…";
  st.style.color = "#374151";

  try{
    const res = await chrome.runtime.sendMessage({type:"ccp_test_device", payload:{device_token: tok}});
    if (res && res.ok){
      st.textContent = "OK";
      st.style.color = "#059669";
      return {ok:true};
    }else{
      st.textContent = (res && res.error) ? res.error : "Lỗi";
      st.style.color = "#dc2626";
      return {ok:false, error: res && res.error};
    }
  }catch(e){
    st.textContent = e.message || String(e);
    st.style.color = "#dc2626";
    return {ok:false, error: e.message || String(e)};
  }
}

$("btnTestAll").addEventListener("click", async () => {
  const rows = Array.from(document.querySelectorAll("#devRows tr"));
  if (!rows.length){ alert("Chưa có thiết bị."); return; }
  let okAll = true;
  for (let i=0;i<rows.length;i++){
    const r = await testOneDevice(i);
    if (!r.ok) okAll = false;
  }
  alert(okAll ? "Tất cả thiết bị OK." : "Có thiết bị lỗi. Kiểm tra lại giúp bạn.");
});

$("btnSave").addEventListener("click", async (ev) => {
  ev.preventDefault();
  const devices = collectDevices();
  if (!devices.length){
    alert("Chưa có thiết bị để lưu.");
    return;
  }
  // Validate before save
  let okAll = true;
  for (let i=0;i<devices.length;i++){
    const r = await testOneDevice(i);
    if (!r.ok) okAll = false;
  }
  if (!okAll){
    if (!confirm("Một số thiết bị lỗi. Bạn vẫn muốn lưu?")) return;
  }
  const base = $("base_origin").value.trim() || DEFAULT_BASE;
  const user = $("auth_user").value.trim();
  const autoCall = $("auto_call").checked;
  await chrome.storage.sync.set({
    [STORE.BASE]: base, [STORE.USER]: user,
    [STORE.DEVICES]: devices, [STORE.AUTOCALL]: autoCall
  });
  alert("Đã lưu.");
  chrome.runtime.sendMessage({type:"ccp_refresh_devices"}).catch(()=>{});
});

$("btnClear").addEventListener("click", async () => {
  await chrome.storage.sync.set({[STORE.XTOKEN]:""});
  $("loginStatus").textContent = "Đã xoá token.";
});

$("btnLogin").addEventListener("click", async () => {
  const base = $("base_origin").value.trim() || DEFAULT_BASE;
  const user = $("auth_user").value.trim();
  const pass = $("auth_pass").value;
  $("loginStatus").textContent = "Đang đăng nhập...";
  try{
    const origin = (()=>{ try{return new URL(base).origin;}catch(_){return base.replace(/\/+$/,"");} })();
    const variants = [`${origin}/auth/login`, `${origin}/api/auth/login`];
    let last;
    for (const url of variants){
      try{
        const res = await fetch(url, { method:"POST", headers: {"Content-Type":"application/json"}, body: JSON.stringify({user, password: pass})});
        if (!res.ok) throw new Error(`HTTP ${res.status}: ${await res.text()}`);
        const data = await res.json();
        const xt = data.token || data.x_token;
        if (!xt) throw new Error("Không nhận được token.");
        await chrome.storage.sync.set({[STORE.USER]: user, [STORE.XTOKEN]: xt});
        $("loginStatus").textContent = "Đăng nhập thành công.";
        return;
      }catch(e){ last = e; }
    }
    throw last || new Error("Đăng nhập thất bại");
  }catch(e){
    $("loginStatus").textContent = "Lỗi: " + (e.message||e);
  }
});


// ===== CHỈ-DÙNG /fcm_tokens.json để tải thiết bị =====
async function fetchDevicesFromServer(base, user){
  const origin = (()=>{ try{ return new URL(base).origin; }catch(_){ return base.replace(/\/+$/,""); } })();
  const url = `${origin}/fcm_tokens.json`;
  const r = await fetch(url, {cache: "no-store"});
  if (!r.ok) throw new Error(`Không tải được /fcm_tokens.json (HTTP ${r.status})`);
  const map = await r.json();
  const out = [];
  for (const [token, item] of Object.entries(map || {})){
    if (user && (item?.user||"") !== user) continue;
    out.push({ device_token: token, name: (item?.device_name||"").trim(), updated: item?.updated || null });
  }
  return out;
}

function renderServerBox(list){
  const box = document.getElementById("serverBox");
  if (!box) return;
  if (!list || !list.length){
    box.style.display = "block";
    box.innerHTML = `<div class="muted">Không thấy thiết bị nào cho user hiện tại.</div>`;
    return;
  }
  const rows = list.map((d,i)=>`
    <tr>
      <td><input type="checkbox" class="pick" data-i="${i}" checked></td>
      <td><input type="text" class="sv-name"  data-i="${i}" value="${d.name||""}"></td>
      <td><input type="text" class="sv-token" data-i="${i}" value="${d.device_token||""}"></td>
      <td><small>${d.updated ? (new Date(d.updated)).toLocaleString() : ""}</small></td>
    </tr>`).join("");
  box.style.display = "block";
  box.innerHTML = `
    <div style="border:1px dashed #d1d5db;border-radius:10px;padding:10px;">
      <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:6px;">
        <b>Thiết bị từ /fcm_tokens.json</b>
        <div>
          <button id="btnPickAll" class="btn ghost">Chọn tất cả</button>
          <button id="btnHideSrv" class="btn ghost">Ẩn</button>
        </div>
      </div>
      <table style="width:100%;border-collapse:separate;border-spacing:0 6px;">
        <thead><tr><th style="width:3rem"></th><th style="width:40%">Tên hiển thị</th><th>Device Token</th><th>Cập nhật</th></tr></thead>
        <tbody>${rows}</tbody>
      </table>
      <div style="margin-top:8px;">
        <button id="btnMergeSrv" class="btn">Thêm mục đã chọn</button>
      </div>
    </div>
  `;
  document.getElementById("btnPickAll").onclick = ()=>{
    document.querySelectorAll("#serverBox .pick").forEach(chk => chk.checked = true);
  };
  document.getElementById("btnHideSrv").onclick = ()=>{ box.style.display = "none"; };
  document.getElementById("btnMergeSrv").onclick = ()=>{
    const picks = Array.from(document.querySelectorAll("#serverBox .pick"));
    const toAdd = [];
    picks.forEach(chk => {
      if (!chk.checked) return;
      const i = +chk.dataset.i;
      const name = document.querySelector(`.sv-name[data-i="${i}"]`)?.value?.trim() || "";
      const tok  = document.querySelector(`.sv-token[data-i="${i}"]`)?.value?.trim() || "";
      if (tok) toAdd.push({name, device_token: tok});
    });
    if (!toAdd.length){ alert("Chưa chọn thiết bị nào."); return; }
    const cur = collectDevices();
    for (const it of toAdd){
      const j = cur.findIndex(x => (x.device_token||x.token) === it.device_token);
      if (j >= 0) cur[j] = it; else cur.push(it);
    }
    renderDevices(cur);
    box.style.display = "none";
  };
}


document.addEventListener("DOMContentLoaded", load);

document.addEventListener("DOMContentLoaded", () => {
  const btn = document.getElementById("btnLoadServer");
  if (!btn) return;
  btn.addEventListener("click", async () => {
    const base = (document.getElementById("base")?.value || document.getElementById("base_origin")?.value || "").trim();
    const user = (document.getElementById("user")?.value || document.getElementById("auth_user")?.value || "").trim();
    if (!base){ alert("Nhập Server trước."); return; }
    try{
      const list = await fetchDevicesFromServer(base, user);
      renderServerBox(list);
    }catch(e){
      alert("Không tải được danh sách từ /fcm_tokens.json: " + (e.message||e));
    }
  });
});
